<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Game;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str; // استيراد مهم لتوليد الكود العشوائي

class GameController extends Controller
{
    /**
     * عرض قائمة الألعاب الموجودة.
     */
    public function index()
    {
        // بما أنها لعبتان فقط، نستخدم all() بدلاً من paginate()
        $games = Game::all(); 
        return view('admin.games.index', compact('games'));
    }

    /**
     * عرض صفحة تعديل صورة اللعبة.
     */
    public function edit(Game $game)
    {
        return view('admin.games.edit', compact('game'));
    }

    /**
     * تحديث صورة اللعبة في قاعدة البيانات.
     */
    public function update(Request $request, Game $game)
    {
        $request->validate([
            'image' => 'required|image|mimes:jpeg,png,jpg,webp|max:2048',
        ]);

        // حذف الصورة القديمة إذا كانت موجودة
        if ($game->image_path) {
            Storage::disk('public')->delete($game->image_path);
        }

        // رفع الصورة الجديدة وتخزين مسارها
        $path = $request->file('image')->store('game_images', 'public');
        
        $game->update(['image_path' => $path]);

        return redirect()->route('admin.games.index')->with('success', 'تم تحديث صورة اللعبة بنجاح.');
    }

    /**
     * ✅ دالة جديدة: إنشاء جلسة لعب مشتركة جديدة
     */
    public function createSharedSession(Request $request)
    {
        $validator = Validator::make($request->all(), ['game_id' => 'required|exists:games,id']);
        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        $user = Auth::user();
        $game = Game::find($request->game_id);

        // إنشاء جلسة جديدة بنوع "مشترك" وحالة "في الانتظار"
        $session = GameSession::create([
            'game_id' => $game->id,
            'user_id' => $user->id, // اللاعب الأول هو منشئ الجلسة
            'total_questions' => $game->max_questions,
            'session_type' => 'shared',
            'status' => 'waiting', // في انتظار اللاعب الثاني
            'join_code' => $this->generateUniqueJoinCode(), // توليد كود فريد
        ]);

        return response()->json([
            'success' => true,
            'message' => 'تم إنشاء الجلسة، شارك الكود مع شريكك.',
            'data' => ['session' => $session]
        ], 201);
    }

    /**
     * ✅ دالة جديدة: الانضمام إلى جلسة لعب مشتركة
     */
    public function joinSharedSession(Request $request)
    {
        $validator = Validator::make($request->all(), ['join_code' => 'required|string|exists:game_sessions,join_code']);
        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => 'الكود الذي أدخلته غير صحيح.'], 404);
        }

        $user = Auth::user();
        $joinCode = $request->join_code;

        // البحث عن الجلسة باستخدام الكود
        $session = GameSession::where('join_code', $joinCode)->first();

        // التحقق من أن الجلسة ما زالت في الانتظار
        if ($session->status !== 'waiting') {
            return response()->json(['success' => false, 'message' => 'هذه اللعبة قد بدأت بالفعل أو انتهت.'], 403);
        }
        
        // التحقق من أن اللاعب لا ينضم إلى جلسته الخاصة
        if ($session->user_id === $user->id) {
            return response()->json(['success' => false, 'message' => 'لا يمكنك الانضمام إلى لعبة قمت أنت بإنشائها.'], 403);
        }

        // تحديث الجلسة بمعلومات اللاعب الثاني وتغيير حالتها إلى "نشطة"
        $session->partner_id = $user->id;
        $session->status = 'active'; // الآن اللعبة جاهزة للبدء
        $session->save();

        // TODO: في الخطوة التالية، سنقوم بإرسال إشعار لحظي للاعب الأول هنا

        return response()->json([
            'success' => true,
            'message' => 'تم الانضمام بنجاح! اللعبة ستبدأ الآن.',
            'data' => ['session' => $session]
        ]);
    }
    
    /**
     * ✅ دالة مساعدة: لتوليد كود انضمام فريد
     */
    private function generateUniqueJoinCode()
    {
        do {
            // توليد كود عشوائي من 6 أحرف كبيرة وأرقام
            $code = strtoupper(Str::random(6));
        } while (GameSession::where('join_code', $code)->exists()); // التحقق من أن الكود غير مستخدم

        return $code;
    }


}