<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\QuestionCategory;
use Illuminate\Http\Request;

class QuestionCategoryController extends Controller
{
    /**
     * عرض صفحة تحتوي على كل الفئات.
     */
    public function index()
    {
        $categories = QuestionCategory::withCount('questions')->latest()->paginate(15);
        return view('admin.question-categories.index', compact('categories'));
    }

    /**
     * عرض نموذج إضافة فئة جديدة.
     */
    public function create()
    {
        return view('admin.question-categories.create');
    }

    /**
     * تخزين فئة جديدة في قاعدة البيانات.
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:100|unique:question_categories,name',
            'description' => 'nullable|string|max:255',
        ]);

        QuestionCategory::create($validatedData);

        return redirect()->route('admin.question-categories.index')
                         ->with('success', 'تمت إضافة الفئة بنجاح.');
    }

    /**
     * عرض نموذج تعديل الفئة.
     */
    public function edit(QuestionCategory $questionCategory)
    {
        return view('admin.question-categories.edit', compact('questionCategory'));
    }

    /**
     * تحديث بيانات الفئة في قاعدة البيانات.
     */
    public function update(Request $request, QuestionCategory $questionCategory)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:100|unique:question_categories,name,' . $questionCategory->id,
            'description' => 'nullable|string|max:255',
        ]);

        $questionCategory->update($validatedData);

        return redirect()->route('admin.question-categories.index')
                         ->with('success', 'تم تحديث الفئة بنجاح.');
    }

    /**
     * حذف فئة من قاعدة البيانات.
     */
    public function destroy(QuestionCategory $questionCategory)
    {
        if ($questionCategory->questions()->count() > 0) {
            return redirect()->route('admin.question-categories.index')
                             ->with('error', 'لا يمكن حذف الفئة لأنها تحتوي على أسئلة مرتبطة بها.');
        }

        $questionCategory->delete();

        return redirect()->route('admin.question-categories.index')
                         ->with('success', 'تم حذف الفئة بنجاح.');
    }
}