<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Question;
use App\Models\QuestionCategory;
use Illuminate\Support\Facades\Validator;

class QuestionController extends Controller
{
    // عرض قائمة الأسئلة
    public function index(Request $request)
    {
        $query = Question::with('category');
        
        // فلترة حسب الفئة
        if ($request->filled('category')) {
            $query->where('question_category_id', $request->category);
        }
        
        // بحث في نص السؤال
        if ($request->filled('search')) {
            $query->where('question_text', 'like', '%' . $request->search . '%');
        }
        
        $questions = $query->latest()->paginate(20);
        $categories = QuestionCategory::all();
        
        return view('admin.questions.index', compact('questions', 'categories'));
    }

    // عرض تفاصيل سؤال محدد
    public function show(Question $question)
    {
        $question->load('category');
        return view('admin.questions.show', compact('question'));
    }

    // عرض صفحة إنشاء سؤال جديد
    public function create()
    {
        $categories = QuestionCategory::all();
        return view('admin.questions.create', compact('categories'));
    }

    // حفظ سؤال جديد
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'question_text' => 'required|string|max:500',
            'question_category_id' => 'required|exists:question_categories,id',
            'question_type' => 'required|in:multiple_choice,text_input',
            'options' => 'required_if:question_type,multiple_choice|array|min:2',
            'options.*.option' => 'required_with:options|string|max:200',
            'options.*.points' => 'nullable|integer|min:1|max:20',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $questionData = [
            'question_text' => $request->question_text,
            'question_category_id' => $request->question_category_id,
            'correct_answer' => null, // للأسئلة الجديدة
        ];

        // معالجة الخيارات
        if ($request->question_type === 'multiple_choice' && $request->options) {
            $options = [];
            foreach ($request->options as $option) {
                if (!empty($option['option'])) {
                    $optionData = ['option' => $option['option']];
                    
                    // إضافة النقاط إذا كانت موجودة (للأسئلة من نوع compatibility)
                    if (!empty($option['points'])) {
                        $optionData['points'] = (int)$option['points'];
                    }
                    
                    $options[] = $optionData;
                }
            }
            $questionData['options'] = json_encode($options);
        } else {
            // للأسئلة النصية - لا خيارات
            $questionData['options'] = null;
        }

        Question::create($questionData);

        return redirect()->route('admin.questions.index')
            ->with('success', 'تم إنشاء السؤال بنجاح');
    }

    // عرض صفحة تعديل سؤال
    public function edit(Question $question)
    {
        $categories = QuestionCategory::all();
        $question->load('category');
        
        // ✅ الحل النهائي: هذا الكود يتعامل مع كل الحالات الممكنة
        $options = [];
        $rawOptions = $question->options;

        if (is_array($rawOptions)) {
            // الحالة الأولى: إذا كانت مصفوفة بالفعل (casting يعمل)
            $options = $rawOptions;
        } elseif (is_string($rawOptions)) {
            // الحالة الثانية: إذا كانت نص JSON (casting لا يعمل لسبب ما)
            $decoded = json_decode($rawOptions, true);
            if (is_array($decoded)) {
                $options = $decoded;
            }
        }
        // في الحالات الأخرى (مثل null)، ستبقى $options مصفوفة فارغة، وهذا آمن.

        return view('admin.questions.edit', compact('question', 'categories', 'options'));
    }
    // تحديث سؤال موجود
    public function update(Request $request, Question $question)
    {
        $validator = Validator::make($request->all(), [
            'question_text' => 'required|string|max:500',
            'question_category_id' => 'required|exists:question_categories,id',
            'question_type' => 'required|in:multiple_choice,text_input',
            'options' => 'required_if:question_type,multiple_choice|array|min:2',
            'options.*.option' => 'required_with:options|string|max:200',
            'options.*.points' => 'nullable|integer|min:1|max:20',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $questionData = [
            'question_text' => $request->question_text,
            'question_category_id' => $request->question_category_id,
        ];

        // معالجة الخيارات
        if ($request->question_type === 'multiple_choice' && $request->options) {
            $options = [];
            foreach ($request->options as $option) {
                if (!empty($option['option'])) {
                    $optionData = ['option' => $option['option']];
                    
                    if (!empty($option['points'])) {
                        $optionData['points'] = (int)$option['points'];
                    }
                    
                    $options[] = $optionData;
                }
            }
            $questionData['options'] = json_encode($options);
        } else {
            $questionData['options'] = null;
        }

        $question->update($questionData);

        return redirect()->route('admin.questions.index')
            ->with('success', 'تم تحديث السؤال بنجاح');
    }

    // حذف سؤال
    public function destroy(Question $question)
    {
        // التحقق من وجود إجابات مرتبطة بهذا السؤال
        $hasAnswers = \App\Models\GameAnswer::where('question_id', $question->id)->exists();
        
        if ($hasAnswers) {
            return redirect()->route('admin.questions.index')
                ->with('error', 'لا يمكن حذف السؤال لأنه يحتوي على إجابات من المستخدمين');
        }
        
        $question->delete();

        return redirect()->route('admin.questions.index')
            ->with('success', 'تم حذف السؤال بنجاح');
    }

    // إنشاء فئة جديدة (AJAX)
    public function createCategory(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:100|unique:question_categories,name',
            'description' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()]);
        }

        $category = QuestionCategory::create([
            'name' => $request->name,
            'description' => $request->description,
        ]);

        return response()->json([
            'success' => true, 
            'category' => $category,
            'message' => 'تم إنشاء الفئة بنجاح'
        ]);
    }
}