<?php

namespace App\Http\Controllers;

use App\Events\PartnerJoined;
use App\Models\Game;
use App\Models\GameSession;
use App\Models\Question;
use App\Models\GameAnswer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class GameController extends Controller
{
    /**
     * دالة لجلب كل الألعاب للصفحة الرئيسية
     */
    public function index()
    {
        try {
            $games = Game::where('is_active', true)->get();
            return response()->json(['success' => true, 'data' => ['games' => $games]]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'حدث خطأ في تحميل الألعاب', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * دالة لجلب إحصائيات المستخدم للصفحة الرئيسية
     */
    public function getUserStats()
    {
        try {
            $user = Auth::user();
            return response()->json([
                'success' => true,
                'data' => [
                    'stats' => [
                        'total_games' => $user->getGamesPlayedCount(),
                        'total_score' => $user->getTotalScore(),
                        'achievements_count' => 0, // سنبنيها لاحقاً
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'حدث خطأ في تحميل الإحصائيات', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * دالة لبدء جلسة لعب جديدة
     */
    public function startSession(Request $request)
    {
        $validator = Validator::make($request->all(), ['game_id' => 'required|exists:games,id']);
        if ($validator->fails()) { return response()->json(['success' => false, 'errors' => $validator->errors()], 422); }

        $user = Auth::user();
        $game = Game::find($request->game_id);

        $session = GameSession::create([
            'game_id' => $game->id,
            'user_id' => $user->id,
            'session_type' => 'single', // تحديد النوع كفردي
            'status' => 'active', // تبدأ اللعبة فوراً
            'total_questions' => $game->max_questions,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'تم بدء اللعبة بنجاح!',
            'data' => [ 'session' => $session, 'game' => $game ]
        ], 201);
    }

    /**
     * دالة لجلب أسئلة جلسة اللعب
     */
    public function getSessionQuestions(GameSession $session)
    {
        $currentUserId = Auth::id();
        
        // تحديث شرط الأمان هنا
        if ($session->user_id !== $currentUserId && $session->partner_id !== $currentUserId) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }
        
        $game = $session->game;
        $questionsQuery = Question::query();
        if ($game->type === 'compatibility') {
            $questionsQuery->whereHas('category', function($query) { $query->where('name', 'أسئلة التوافق'); });
        } else {
            $questionsQuery->whereHas('category', function($query) { $query->where('name', 'أسئلة عامة'); });
        }
        $questions = $questionsQuery->inRandomOrder()->limit($game->max_questions)->get();
        return response()->json(['success' => true, 'data' => ['questions' => $questions]]);
    }

    /**
     * دالة جديدة: إنشاء جلسة لعب مشتركة جديدة
     */
    public function createSharedSession(Request $request)
    {
        $validator = Validator::make($request->all(), ['game_id' => 'required|exists:games,id']);
        if ($validator->fails()) { return response()->json(['success' => false, 'errors' => $validator->errors()], 422); }

        $user = Auth::user();
        $game = Game::find($request->game_id);
        
        // إنشاء كود فريد وعشوائي
        do {
            $joinCode = Str::upper(Str::random(6));
        } while (GameSession::where('join_code', $joinCode)->where('status', '!=', 'completed')->exists());

        $session = GameSession::create([
            'game_id' => $game->id,
            'user_id' => $user->id,
            'join_code' => $joinCode,
            'session_type' => 'shared',
            'status' => 'waiting', // في انتظار اللاعب الثاني
            'total_questions' => $game->max_questions,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'تم إنشاء الجلسة بنجاح، شارك الكود مع شريكك.',
            'data' => ['session' => $session]
        ], 201);
    }

    /**
     * دالة لانضمام اللاعب الثاني إلى جلسة مشتركة.
     */
    public function joinSharedSession(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'join_code' => 'required|string|exists:game_sessions,join_code',
        ]);
        
        if ($validator->fails()) { 
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422); 
        }

        $user = Auth::user();
        $joinCode = strtoupper($request->join_code);

        $session = GameSession::where('join_code', $request->join_code)
                                ->where('session_type', 'shared')
                                ->where('status', 'waiting')
                                ->first();
        if (!$session) {
            return response()->json([
                'success' => false, 
                'message' => 'الكود غير صالح أو الجلسة لم تعد متاحة'
            ], 404);
        }
        
        if ($session->user_id == $user->id) {
            return response()->json([
                'success' => false, 
                'message' => 'لا يمكنك الانضمام إلى لعبتك'
            ], 400);
        }

        // تحديث الجلسة
        $session->partner_id = $user->id;
        $session->status = 'active';
        $session->save();
        
        try {
            broadcast(new \App\Events\PartnerJoined($session));
        } catch (\Exception $e) {
            \Log::error("❌ Event broadcast failed: " . $e->getMessage());
        }
        
        return response()->json([
            'success' => true,
            'message' => 'تم الانضمام بنجاح!',
            'data' => [
                'session' => $session->fresh()->load(['user', 'partner', 'game']),
                'game' => $session->game,
            ]
        ]);
    }

    /**
     * دالة جديدة للتحقق من حالة الجلسة
     */
    public function getSessionStatus($sessionId)
    {
        try {
            $user = Auth::user();
            
            $session = GameSession::with(['user:id,name,email', 'partner:id,name,email', 'game:id,name,type'])
                ->select('id', 'join_code', 'session_type', 'game_id', 'user_id', 'partner_id', 'status', 'score', 'total_questions', 'correct_answers', 'started_at', 'is_completed', 'completed_at', 'created_at', 'updated_at')
                ->find($sessionId);
            
            if (!$session) {
                return response()->json([
                    'success' => false,
                    'message' => 'الجلسة غير موجودة'
                ], 404);
            }
            
            if (!$session->hasPlayer($user->id)) {
                return response()->json([
                    'success' => false,
                    'message' => 'غير مصرح'
                ], 403);
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'session' => $session,
                    'game' => $session->game
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطأ'
            ], 500);
        }
    }
    
    /**
     * دالة لحفظ إجابة اللاعب
     */
    public function submitAnswer(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'game_session_id' => 'required|exists:game_sessions,id',
            'question_id' => 'required|exists:questions,id',
            'answer' => 'required|string',
        ]);
        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }
        GameAnswer::create([
            'game_session_id' => $request->game_session_id,
            'question_id' => $request->question_id,
            'user_id' => Auth::id(),
            'answer' => $request->answer,
        ]);
        return response()->json(['success' => true, 'message' => 'تم حفظ إجابتك بنجاح']);
    }

    /**
     * دالة لمقارنة إجابات اللاعبين لنفس السؤال
     */
    public function compareAnswers(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'game_session_id' => 'required|exists:game_sessions,id',
            'question_id' => 'required|exists:questions,id',
        ]);
        if ($validator->fails()) { return response()->json(['success' => false, 'errors' => $validator->errors()], 422); }
        
        $answers = GameAnswer::where('game_session_id', $request->game_session_id)
                                ->where('question_id', $request->question_id)
                                ->latest()->take(2)->get();
        
        if ($answers->count() < 2) {
            return response()->json(['success' => false, 'message' => 'لم يتم العثور على إجابات كافية للمقارنة'], 404);
        }

        $player1_answer = $answers[1]->answer;
        $player2_answer = $answers[0]->answer;

        $isMatch = (strtolower(trim($player1_answer)) == strtolower(trim($player2_answer)));
        $points = $isMatch ? 10 : 0;

        $session = GameSession::find($request->game_session_id);
        $session->score += $points;
        if ($isMatch) { $session->correct_answers += 1; }
        $session->save();

        return response()->json(['success' => true, 'data' => [ 'is_match' => $isMatch, 'points_earned' => $points, 'player1_answer' => $player1_answer, 'player2_answer' => $player2_answer ]]);
    }
    
    /**
     * دالة لإنهاء جلسة اللعب
     */
    public function endSession(GameSession $session)
    {
        if (! $session->hasPlayer(Auth::id())) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }
        $session->status = 'completed';
        $session->completed_at = now();
        $session->save();
        return response()->json(['success' => true, 'message' => 'Game session completed.']);
    }

    /**
     * الدالة الجديدة والمهمة لحساب التوافق
     */
    public function calculateCompatibility(GameSession $session)
    {
        if (! $session->hasPlayer(Auth::id())) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $answers = GameAnswer::where('game_session_id', $session->id)->get();
        $totalPoints = 0;
        
        foreach ($answers as $answer) {
            $question = Question::find($answer->question_id);
            
            $options = $question->options;
            
            if (is_string($options)) {
                $options = json_decode($options, true);
            }
            
            if (is_array($options)) {
                foreach ($options as $option) {
                    if (isset($option['option']) && isset($option['points']) && $option['option'] === $answer->answer) {
                        $totalPoints += $option['points'];
                        break;
                    }
                }
            }
        }

        $questionCount = $answers->unique('question_id')->count();
        $maxPossiblePoints = $questionCount * 15; // أعلى قيمة للخيار هي 15
        $compatibilityScore = ($maxPossiblePoints > 0) ? round(($totalPoints / $maxPossiblePoints) * 100) : 0;
        
        $session->score = $compatibilityScore;
        $session->save();

        $message = $this->getCompatibilityMessage($compatibilityScore);

        return response()->json([
            'success' => true,
            'data' => [
                'score' => $compatibilityScore,
                'message' => $message,
                'total_points' => $totalPoints,
                'max_points' => $maxPossiblePoints,
            ]
        ]);
    }

    /**
     * دالة جديدة لإنشاء رسائل مخصصة حسب النسبة
     */
    private function getCompatibilityMessage($score)
    {
        if ($score >= 90) {
            return "توافق استثنائي! أنتما متفقان في معظم الأمور الأساسية.";
        } elseif ($score >= 75) {
            return "توافق ممتاز! لديكما أسس قوية للعلاقة مع بعض الاختلافات المثيرة.";
        } elseif ($score >= 60) {
            return "توافق جيد! لديكما نقاط التقاء واضحة مع مساحة للنمو.";
        } elseif ($score >= 45) {
            return "توافق متوسط! هناك إمكانيات للتفاهم مع الحاجة للمزيد من التواصل.";
        } elseif ($score >= 30) {
            return "توافق منخفض! قد تحتاجان للعمل على تفهم وجهات نظر بعضكما.";
        } else {
            return "اختلافات كبيرة! هذا ليس سيئاً - الاختلافات يمكن أن تكون مكملة.";
        }
    }
    
    // ✅✅✅ ---- بداية الدالة الجديدة التي تمت إضافتها ---- ✅✅✅
    /**
     * الدالة الجديدة للتحقق من إجابات كلا اللاعبين على سؤال معين
     *
     * @param \App\Models\GameSession $session
     * @param \App\Models\Question $question
     * @return \Illuminate\Http\JsonResponse
     */
    public function getQuestionStatus(GameSession $session, Question $question)
    {
        // التحقق من أن المستخدم الحالي هو جزء من اللعبة
        if (! $session->hasPlayer(Auth::id())) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        // عد الإجابات المسجلة لهذا السؤال في هذه الجلسة
        $answersCount = GameAnswer::where('game_session_id', $session->id)
                                  ->where('question_id', $question->id)
                                  ->count();

        // إذا كان عدد الإجابات أقل من 2، فهذا يعني أننا ما زلنا ننتظر
        if ($answersCount < 2) {
            return response()->json([
                'success' => false,
                'message' => 'Waiting for partner to answer.',
                'data' => ['answers_count' => $answersCount]
            ], 202); // 202 Accepted, but not yet processed
        }

        // إذا كان هناك إجابتان، قم بمقارنتهما وإرجاع النتيجة
        $answers = GameAnswer::where('game_session_id', $session->id)
                               ->where('question_id', $question->id)
                               ->latest()->take(2)->get();
        
        // تأكد من وجود إجابتين قبل الوصول إلى الفهرس
        if ($answers->count() < 2) {
             return response()->json([
                'success' => false,
                'message' => 'Could not retrieve both answers for comparison.'
            ], 500);
        }

        $player1_answer = $answers[1]->answer; // الإجابة الأقدم
        $player2_answer = $answers[0]->answer; // الإجابة الأحدث

        $isMatch = (strtolower(trim($player1_answer)) == strtolower(trim($player2_answer)));

        return response()->json([
            'success' => true,
            'message' => 'Both players have answered.',
            'data' => [
                'is_match' => $isMatch,
                'player1_answer' => $player1_answer,
                'player2_answer' => $player2_answer
            ]
        ]);
    }
    // ✅✅✅ ---- نهاية الدالة الجديدة ---- ✅✅✅

}