<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class GameSession extends Model
{
    use HasFactory;

    protected $fillable = [
        'game_id',
        'user_id',
        'partner_id',
        'session_type',
        'status',
        'join_code',
        'total_questions',
        'score',
        'correct_answers',
        'completed_at'
    ];

    protected $casts = [
        'completed_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * العلاقة مع اللعبة
     */
    public function game()
    {
        return $this->belongsTo(Game::class);
    }

    /**
     * العلاقة مع اللاعب الأول (منشئ الجلسة)
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * العلاقة مع اللاعب الثاني (الشريك)
     */
    public function partner()
    {
        return $this->belongsTo(User::class, 'partner_id');
    }

    /**
     * العلاقة مع الإجابات
     */
    public function answers()
    {
        return $this->hasMany(GameAnswer::class);
    }

    /**
     * نطاق للجلسات النشطة
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * نطاق للجلسات المكتملة
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * نطاق للجلسات المنتظرة
     */
    public function scopeWaiting($query)
    {
        return $query->where('status', 'waiting');
    }

    /**
     * نطاق للجلسات المشتركة
     */
    public function scopeShared($query)
    {
        return $query->where('session_type', 'shared');
    }

    /**
     * نطاق للجلسات الفردية
     */
    public function scopeSingle($query)
    {
        return $query->where('session_type', 'single');
    }

    /**
     * التحقق من أن المستخدم جزء من هذه الجلسة
     */
    public function hasPlayer($userId)
    {
        return $this->user_id == $userId || $this->partner_id == $userId;
    }

    /**
     * الحصول على الشريك للمستخدم المحدد
     */
    public function getPartnerFor($userId)
    {
        if ($this->user_id == $userId) {
            return $this->partner;
        } elseif ($this->partner_id == $userId) {
            return $this->user;
        }
        return null;
    }

    /**
     * التحقق من اكتمال الجلسة
     */
    public function isCompleted()
    {
        return $this->status === 'completed';
    }

    /**
     * التحقق من أن الجلسة نشطة
     */
    public function isActive()
    {
        return $this->status === 'active';
    }

    /**
     * التحقق من أن الجلسة في انتظار اللاعب الثاني
     */
    public function isWaiting()
    {
        return $this->status === 'waiting';
    }
}